<?php
include 'includes/config.php';
include 'includes/functions.php';

if (!isLoggedIn()) {
    redirect('index.php');
}

$form = null;
$questions = [];

if (isset($_GET['edit'])) {
    $form_id = $_GET['edit'];
    $form = getForm($pdo, $form_id);
    $questions = getQuestions($pdo, $form_id);
}

if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    if (isset($_POST['save_form'])) {
        $title = $_POST['title'];
        $description = $_POST['description'];
        
        if (isset($_POST['form_id'])) {
            // Editar formulário existente
            $form_id = $_POST['form_id'];
            $stmt = $pdo->prepare("UPDATE forms SET title = ?, description = ? WHERE id = ?");
            $stmt->execute([$title, $description, $form_id]);
        } else {
            // Criar novo formulário
            $stmt = $pdo->prepare("INSERT INTO forms (title, description, created_by) VALUES (?, ?, ?)");
            $stmt->execute([$title, $description, $_SESSION['user_id']]);
            $form_id = $pdo->lastInsertId();
        }
        
        // Salvar perguntas
        if (isset($_POST['questions'])) {
            // Remover perguntas antigas
            $stmt = $pdo->prepare("DELETE FROM questions WHERE form_id = ?");
            $stmt->execute([$form_id]);
            
            foreach ($_POST['questions'] as $index => $question_data) {
                $question_text = $question_data['text'];
                $question_type = $question_data['type'];
                
                $stmt = $pdo->prepare("INSERT INTO questions (form_id, question_text, question_type, question_order) VALUES (?, ?, ?, ?)");
                $stmt->execute([$form_id, $question_text, $question_type, $index]);
                $question_id = $pdo->lastInsertId();
                
                // Salvar opções se for radio, checkbox ou select
                if (in_array($question_type, ['radio', 'checkbox', 'select']) && isset($question_data['options'])) {
                    foreach ($question_data['options'] as $option_text) {
                        if (!empty(trim($option_text))) {
                            $stmt = $pdo->prepare("INSERT INTO question_options (question_id, option_text) VALUES (?, ?)");
                            $stmt->execute([$question_id, trim($option_text)]);
                        }
                    }
                }
            }
        }
        
        redirect('dashboard.php');
    }
}
?>

<!DOCTYPE html>
<html lang="pt-BR">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo isset($form) ? 'Editar' : 'Criar'; ?> Formulário - FormBuilder</title>
    <link rel="stylesheet" href="assets/css/style.css">
    <script>
        let questionCount = <?php echo count($questions); ?>;
        
        function addQuestion() {
            questionCount++;
            const container = document.getElementById('questions-container');
            
            const questionHTML = `
                <div class="question-builder" id="question-${questionCount}">
                    <div class="form-group">
                        <label>Pergunta:</label>
                        <input type="text" name="questions[${questionCount}][text]" required>
                    </div>
                    
                    <div class="form-group">
                        <label>Tipo de Resposta:</label>
                        <select name="questions[${questionCount}][type]" onchange="toggleOptions(${questionCount})">
                            <option value="text">Texto</option>
                            <option value="textarea">Texto Longo</option>
                            <option value="radio">Opção Única</option>
                            <option value="checkbox">Múltipla Escolha</option>
                            <option value="select">Menu Suspenso</option>
                        </select>
                    </div>
                    
                    <div class="form-group options-container" id="options-${questionCount}" style="display: none;">
                        <label>Opções (uma por linha):</label>
                        <textarea name="questions[${questionCount}][options]" rows="3" placeholder="Digite cada opção em uma linha"></textarea>
                    </div>
                    
                    <button type="button" class="btn btn-danger" onclick="removeQuestion(${questionCount})">Remover Pergunta</button>
                </div>
            `;
            
            container.insertAdjacentHTML('beforeend', questionHTML);
        }
        
        function toggleOptions(questionId) {
            const select = document.querySelector(`select[name="questions[${questionId}][type]"]`);
            const optionsContainer = document.getElementById(`options-${questionId}`);
            
            if (['radio', 'checkbox', 'select'].includes(select.value)) {
                optionsContainer.style.display = 'block';
            } else {
                optionsContainer.style.display = 'none';
            }
        }
        
        function removeQuestion(questionId) {
            const element = document.getElementById(`question-${questionId}`);
            element.remove();
        }
    </script>
</head>
<body>
    <div class="container">
        <?php include 'includes/header.php'; ?>
        
        <div class="main-content">
            <div class="header">
                <h1><?php echo isset($form) ? 'Editar' : 'Criar'; ?> Formulário</h1>
                <a href="dashboard.php" class="btn btn-secondary">Voltar</a>
            </div>
            
            <form method="POST">
                <?php if (isset($form)): ?>
                    <input type="hidden" name="form_id" value="<?php echo $form['id']; ?>">
                <?php endif; ?>
                
                <div class="card">
                    <h2>Configurações do Formulário</h2>
                    <div class="form-group">
                        <label>Título:</label>
                        <input type="text" name="title" value="<?php echo isset($form) ? htmlspecialchars($form['title']) : ''; ?>" required>
                    </div>
                    
                    <div class="form-group">
                        <label>Descrição:</label>
                        <textarea name="description" rows="3"><?php echo isset($form) ? htmlspecialchars($form['description']) : ''; ?></textarea>
                    </div>
                </div>
                
                <div class="card">
                    <h2>Perguntas</h2>
                    <div id="questions-container">
                        <?php if (!empty($questions)): ?>
                            <?php foreach ($questions as $index => $question): ?>
                                <div class="question-builder" id="question-<?php echo $index; ?>">
                                    <div class="form-group">
                                        <label>Pergunta:</label>
                                        <input type="text" name="questions[<?php echo $index; ?>][text]" value="<?php echo htmlspecialchars($question['question_text']); ?>" required>
                                    </div>
                                    
                                    <div class="form-group">
                                        <label>Tipo de Resposta:</label>
                                        <select name="questions[<?php echo $index; ?>][type]" onchange="toggleOptions(<?php echo $index; ?>)">
                                            <option value="text" <?php echo $question['question_type'] == 'text' ? 'selected' : ''; ?>>Texto</option>
                                            <option value="textarea" <?php echo $question['question_type'] == 'textarea' ? 'selected' : ''; ?>>Texto Longo</option>
                                            <option value="radio" <?php echo $question['question_type'] == 'radio' ? 'selected' : ''; ?>>Opção Única</option>
                                            <option value="checkbox" <?php echo $question['question_type'] == 'checkbox' ? 'selected' : ''; ?>>Múltipla Escolha</option>
                                            <option value="select" <?php echo $question['question_type'] == 'select' ? 'selected' : ''; ?>>Menu Suspenso</option>
                                        </select>
                                    </div>
                                    
                                    <div class="form-group options-container" id="options-<?php echo $index; ?>" style="<?php echo in_array($question['question_type'], ['radio', 'checkbox', 'select']) ? 'display: block;' : 'display: none;'; ?>">
                                        <label>Opções (uma por linha):</label>
                                        <?php
                                        $options = getQuestionOptions($pdo, $question['id']);
                                        $options_text = '';
                                        foreach ($options as $option) {
                                            $options_text .= $option['option_text'] . "\n";
                                        }
                                        ?>
                                        <textarea name="questions[<?php echo $index; ?>][options]" rows="3" placeholder="Digite cada opção em uma linha"><?php echo trim($options_text); ?></textarea>
                                    </div>
                                    
                                    <button type="button" class="btn btn-danger" onclick="removeQuestion(<?php echo $index; ?>)">Remover Pergunta</button>
                                </div>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </div>
                    
                    <button type="button" class="btn btn-primary" onclick="addQuestion()">Adicionar Pergunta</button>
                </div>
                
                <button type="submit" name="save_form" class="btn btn-success">Salvar Formulário</button>
            </form>
        </div>
    </div>
</body>
</html>